<?php
// remote_content_manager.php

require_once 'RemoteDB.php';

// 初始化远程数据库连接
#$db = new RemoteDB('content_ai.db', 'http://pan.zgyzty.com/api/AIGC/renzaocao/');

// 配置
define('ITEMS_PER_PAGE', 10);

// 获取内容列表
function getContents($db, $page = 1, $keyword = '', $categoryId = null) {
    $offset = ($page - 1) * ITEMS_PER_PAGE;
    
    // 构建基础查询
    $sql = "SELECT gc.id, gc.question, gc.answer, gc.created_at, 
                   c.id as category_id, c.name as category_name 
            FROM generated_content gc
            LEFT JOIN categories c ON gc.category_id = c.id";
    
    $params = [];
    $where = [];
    
    // 添加搜索条件
    if (!empty($keyword)) {
        $where[] = "(gc.question LIKE ? OR gc.answer LIKE ?)";
        $params[] = "%$keyword%";
        $params[] = "%$keyword%";
    }
    
    // 添加分类筛选
    if ($categoryId) {
        $where[] = "gc.category_id = ?";
        $params[] = $categoryId;
    }
    
    // 组合WHERE子句
    if (!empty($where)) {
        $sql .= " WHERE " . implode(" AND ", $where);
    }
    
    // 添加分页
    $sql .= " ORDER BY gc.created_at DESC LIMIT ? OFFSET ?";
    $params[] = ITEMS_PER_PAGE;
    $params[] = $offset;
    
    // 执行查询
    $contents = $db->query($sql, $params) ?: [];
    
    // 获取总数
    $countSql = "SELECT COUNT(*) as total FROM generated_content gc";
    if (!empty($where)) {
        $countSql .= " WHERE " . implode(" AND ", $where);
    }
    $total = $db->query($countSql, $params)[0]['total'] ?? 0;
    
    // 获取分类列表
    $categories = $db->query("SELECT id, name FROM categories ORDER BY name") ?: [];
    
    return [
        'contents' => $contents,
        'total' => $total,
        'total_pages' => ceil($total / ITEMS_PER_PAGE),
        'current_page' => $page,
        'categories' => $categories,
        'current_category' => $categoryId,
        'keyword' => $keyword
    ];
}

// 获取单个内容
function getContent($db, $id) {
    $sql = "SELECT gc.id, gc.question, gc.answer, gc.created_at, 
                   c.id as category_id, c.name as category_name 
            FROM generated_content gc
            LEFT JOIN categories c ON gc.category_id = c.id
            WHERE gc.id = ?";
    $result = $db->query($sql, [$id]);
    return $result[0] ?? null;
}

// 主控制器
function handleRequest($db) {
    $action = isset($_GET['action']) ? $_GET['action'] : 'list';
    $outputType = isset($_GET['type']) ? $_GET['type'] : 'html';
    
    try {
        switch ($action) {
            case 'list':
                $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
                $keyword = isset($_GET['keyword']) ? trim($_GET['keyword']) : '';
                $categoryId = isset($_GET['category']) ? intval($_GET['category']) : null;
                
                $data = getContents($db, $page, $keyword, $categoryId);
                
                if ($outputType === 'json') {
                    header('Content-Type: application/json');
                    echo json_encode($data);
                } else {
                    displayListPage($data);
                }
                break;
                
            case 'view':
                if (!isset($_GET['id'])) {
                    throw new Exception("缺少ID参数");
                }
                
                $id = intval($_GET['id']);
                $content = getContent($db, $id);
                
                if ($outputType === 'json') {
                    header('Content-Type: application/json');
                    echo json_encode($content ?: ['error' => '内容不存在']);
                } else {
                    displayContentPage($content);
                }
                break;
                
            case 'test':
                // 测试数据库连接
                $testInsert = $db->exec(
                    "INSERT INTO test(title, `key`, `desc`) VALUES(?, ?, ?)", 
                    [date("H:i:s"), date("H:i:s"), date("H:i:s")]
                );
                
                $testResults = $db->query("SELECT * FROM test", []);
                
                echo "<h2>数据库连接测试</h2>";
                echo "<p>插入记录ID: " . ($testInsert !== false ? $testInsert : '失败') . "</p>";
                echo "<h3>测试表内容:</h3>";
                echo "<pre>";
                print_r($testResults);
                echo "</pre>";
                echo "<a href='?action=list'>返回列表</a>";
                break;
                
            default:
                throw new Exception("无效的操作");
        }
    } catch (Exception $e) {
        if ($outputType === 'json') {
            header('Content-Type: application/json');
            echo json_encode(['error' => $e->getMessage()]);
        } else {
            echo "<h2>错误</h2>";
            echo "<p>" . htmlspecialchars($e->getMessage()) . "</p>";
            echo "<a href='?action=list'>返回列表</a>";
        }
    }
}

// 显示列表页
function displayListPage($data) {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>内容管理系统</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 20px; }
            table { width: 100%; border-collapse: collapse; }
            th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            th { background-color: #f2f2f2; }
            .pagination { margin-top: 20px; }
            .search-form { margin-bottom: 20px; }
        </style>
    </head>
    <body>
        <h1>内容管理系统</h1>
        
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>分类</th>
                    <th>问题</th>
                    <th>创建时间</th>
                    <th>操作</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($data['contents'] as $item): ?>
                    <tr>
                        <td><?= htmlspecialchars($item['id']) ?></td>
                        <td><?= htmlspecialchars($item['category_name']) ?></td>
                        <td><?= htmlspecialchars(mb_substr($item['question'], 0, 50, 'UTF-8')) ?></td>
                        <td><?= htmlspecialchars($item['created_at']) ?></td>
                        <td>
                            <a href="?action=view&id=<?= $item['id'] ?>">查看</a>
                            <a href="?action=view&id=<?= $item['id'] ?>&type=json">JSON</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <?php if ($data['total_pages'] > 1): ?>
            <div class="pagination">
                <?php if ($data['current_page'] > 1): ?>
                    <a href="?action=list&page=1&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">首页</a>
                    <a href="?action=list&page=<?= $data['current_page']-1 ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">上一页</a>
                <?php endif; ?>
                
                <?php for ($i = 1; $i <= $data['total_pages']; $i++): ?>
                    <?php if ($i == $data['current_page']): ?>
                        <strong><?= $i ?></strong>
                    <?php else: ?>
                        <a href="?action=list&page=<?= $i ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>"><?= $i ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                
                <?php if ($data['current_page'] < $data['total_pages']): ?>
                    <a href="?action=list&page=<?= $data['current_page']+1 ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">下一页</a>
                    <a href="?action=list&page=<?= $data['total_pages'] ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">末页</a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </body>
    </html>
    <?php
}

// 显示内容页
function displayContentPage($content) {
    if (!$content) {
        echo "<h2>内容不存在</h2>";
        echo "<a href='?action=list'>返回列表</a>";
        return;
    }
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title><?= htmlspecialchars(mb_substr($content['question'], 0, 20, 'UTF-8')) ?></title>
        <style>
            body { font-family: Arial, sans-serif; margin: 20px; }
            .content-box { border: 1px solid #ddd; padding: 15px; margin: 15px 0; }
        </style>
    </head>
    <body>
        <h1><?= htmlspecialchars($content['question']) ?></h1>
        <p>分类: <?= htmlspecialchars($content['category_name']) ?></p>
        <p>创建时间: <?= htmlspecialchars($content['created_at']) ?></p>
        
        <div class="content-box">
            <?= nl2br(htmlspecialchars($content['answer'])) ?>
        </div>
        
        <div>
            <a href="?action=list">返回列表</a> | 
            <a href="?action=view&id=<?= $content['id'] ?>&type=json">查看JSON格式</a>
        </div>
    </body>
    </html>
    <?php
}
$db = new RemoteDB('content_ai.db', 'http://pan.zgyzty.com/api/AIGC/renzaocao/');
// 初始化并运行
handleRequest($db);
?>